const User = require('../model/user');
const OTPModel = require("../model/otp");

const email_templates = require('../config/email_template_ids');
const bcrypt = require("bcrypt");
const jwt = require("jsonwebtoken");
const helper = require('../middleware/_helper');
const mediaUtils = require("../config/unlinkMedia");

const emailCtrl = require('../controllers/email.controller');
const otpGenerator = require('otp-generator');
var moment = require('moment'); // require
const path = require("path");
var fs = require("fs");
const fse = require('fs-extra')

// Create and Save a new User
exports.create = async (req, res) => {

    // Our Add user logic starts here
    try {
        // Get user input
        const { first_name, last_name, email, phone, address_line, country, state, city, zip_code } = req.body;

        // Validate user input
        if (!(email && first_name && last_name)) {
            return res.status(400).send({ "is_error": true, "message": "All input ( Email, first & last name) is required" });
        }

        if(req.files && req.files['profile_photo']){
            const mediaRootPath = "public/uploads/profile_photos/";
            const path =  mediaRootPath + req.files['profile_photo'][0].filename;
            req.body.profile_photo = path ;
        }

        // check if user already exist
        // Validate if user exist in our database
        const oldUser = await User.findOne({ email: req.body.email.toLowerCase() });

        if (oldUser) {
            if(req.files && req.files['profile_photo']){
                const mediaRootPath = "public/uploads/profile_photos/";
                const path =  mediaRootPath + req.files['profile_photo'][0].filename;
                // delete all media from folder
                mediaUtils.unlink(path);
              }
            return res.status(409).send({ "is_error": true, "message": "User Already Exist." });
        }

        const tempPassword = otpGenerator.generate(7);
        console.log(tempPassword,'Temporary Password');
        //Encrypt user password
        encryptedPassword = await bcrypt.hash(tempPassword, 10);

        // Create user in our database
        const user = await User.create({
            first_name,
            last_name,
            email: email.toLowerCase(), // sanitize: convert email to lowercase
            role: req.body.role,
            password: encryptedPassword,
            phone,
            address_line,
            country,
            state,
            city,
            zip_code,
            profile_photo : req.body.profile_photo,
            countryMetaData: JSON.parse(req.body.countryMetaData),
            stateMetaData: JSON.parse(req.body.stateMetaData),
            cityMetaData: JSON.parse(req.body.cityMetaData),
        });

        const emailData = {
            msg: {
                to: email,
                from: 'book@ctn.vacations', // Use the email address or domain you verified above
                template_id: email_templates.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
                dynamic_template_data:{
                  "first_name": user.first_name,
                  "last_name": user.last_name,
                  "password": tempPassword
                }
              },
            userDetails: user
        }
        // send Email to User with OTP
        // emailCtrl.sendEmail(emailData);

        // return new user
        res.status(200).json({ is_error: false, data: user });
    } catch (err) {
        console.log(err);
        if(req.files && req.files['profile_photo']){
            const mediaRootPath = "public/uploads/profile_photos/";
            const path =  mediaRootPath + req.files['profile_photo'][0].filename;
            // delete all media from folder
            mediaUtils.unlink(path);
          }
        res.status(500).json({ is_error: true, data: {}, message:"Error While Creating user.Please try again."+ err});

    }
    // Our Add user logic ends here

};

// Retrieve and return all users from the database.
exports.findAll = async (req, res) => {
    try {
        const user = await User.find({is_admin: false}).select("-password -__v").sort([["updatedAt",-1]]);
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};

// Find a single user with a userId
exports.findOne = async (req, res) => {
    User.findById(req.params.userId).select("-password -__v")
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            res.send({ is_error: false, data: userData });
        }).catch(err => {
            console.log("err", err)
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error retrieving user with id " + req.params.userId
            });
        });
};

// Update a user identified by the userId in the request
exports.update = async (req, res) => {
    console.log("update req.body", req.body);
    console.log("req.params", req.params)
    // Validate Request
    if (!(req.body.email && req.body.first_name && req.body.last_name)) {
        return res.status(400).send({ "is_error": true, "message": "All inputs are required." });
    }
    // check if user already exist
    // Validate if user exist in our database
    const oldUser = await User.find({ email: req.body.email.toLowerCase(), _id: { $ne: req.params.userId } });
    console.log("old user", oldUser)
    if (oldUser.length > 0) {
        if(req.files && req.files['profile_photo']){
            const mediaRootPath = "public/uploads/profile_photos/";
            const path =  mediaRootPath + req.files['profile_photo'][0].filename;
            // delete all media from folder
            mediaUtils.unlink(path);
          }
        return res.status(409).send({ "is_error": true, "message": "User already exist with this email" });
    }
    // Find note and update it with the request body
    await User.findByIdAndUpdate(req.params.userId, {
        first_name: req.body.first_name,
        last_name: req.body.last_name,
        email: req.body.email.toLowerCase(), //sanitize convert to lowercase
        role: req.body.role,
        phone: req.body.phone,
        address_line: req.body.address_line,
        country: req.body.country,
        state: req.body.state,
        city: req.body.city,
        zip_code: req.body.zip_code,
        countryMetaData: JSON.parse(req.body.countryMetaData),
        stateMetaData: JSON.parse(req.body.stateMetaData),
        cityMetaData: JSON.parse(req.body.cityMetaData),
    }, { new: true })
        .then(async response => {
            if(req.files && req.files['profile_photo']){
                const mediaRootPath = "public/uploads/profile_photos/";
                const path =  mediaRootPath + req.files['profile_photo'][0].filename;
                const newprofile_photo = path ;
                // check if photo is already set then delete the media from folder
                if(response.profile_photo  && response.profile_photo != ''){
                    mediaUtils.unlink(response.profile_photo);
                }
                await User.findByIdAndUpdate(req.params.userId, {profile_photo: newprofile_photo}, {new: true}).exec();
              }
            if (!response) {
                if(req.files && req.files['profile_photo']){
                    const mediaRootPath = "public/uploads/profile_photos/";
                    const path =  mediaRootPath + req.files['profile_photo'][0].filename;
                    // delete all media from folder
                    mediaUtils.unlink(path);
                  }
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found."
                });
            }
            res.send({
                is_error: false,
                data: response
            });
        }).catch(err => {
            if(req.files && req.files['profile_photo']){
                const mediaRootPath = "public/uploads/profile_photos/";
                const path =  mediaRootPath + req.files['profile_photo'][0].filename;
                // delete all media from folder
                mediaUtils.unlink(path);
              }
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the account details."
            });
        });
};

// Delete a userData with the specified userId in the request
exports.delete = async (req, res) => {
    
    User.findByIdAndRemove(req.params.userId)
        .then(async userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            res.send({
                is_error: false,
                message: "User deleted successfully!"
            });
        }).catch(err => {
            if (err.kind === 'ObjectId' || err.name === 'NotFound') {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Could not delete User with id " + req.params.userId
            });
        });
};

// User specific APIS

// Update a user identified by the userId in the request
exports.updateMyAccount = async (req, res) => {

    // Validate Request
    if (!(req.body.email && req.body.first_name && req.body.last_name)) {
        return res.status(400).send({ "is_error": true, "message": "All inputs are required." });
    }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    if (loggedInUser && loggedInUser.data && loggedInUser.data.user_id) {
        // Validate if email already exist in our database
        const oldUser = await User.find({ email: req.body.email.toLowerCase(), _id: { $ne: loggedInUser.data.user_id } });
        console.log("old user", oldUser)
        if (oldUser.length > 0) {
            return res.status(409).send({ "is_error": true, "message": "Email not available. Please use some different Email to update your profile details." });
        }
        // Find note and update it with the request body
        await User.findByIdAndUpdate(loggedInUser.data.user_id, {
            first_name: req.body.first_name,
            last_name: req.body.last_name,
            gender: req.body.gender,
            email: req.body.email.toLowerCase(),
            phone: req.body.phone,
            country:  req.body.country,
            state:  req.body.state,
            city:  req.body.city,
            zip_code:  req.body.zip_code,
            address_line:  req.body.address_line,
            countryMetaData:  req.body.countryMetaData,
            stateMetaData:  req.body.stateMetaData,
            cityMetaData:  req.body.cityMetaData,

        }, { new: true })
            .then(userData => {
                if (!userData) {
                    return res.status(404).send({
                        is_error: true,
                        message: "User details not found."
                    });
                }
                res.send({
                    is_error: false,
                    data: userData
                });
            }).catch(err => {
                if (err.kind === 'ObjectId') {
                    return res.status(404).send({
                        is_error: true,
                        message: "User details not found with provided Id."
                    });
                }
                return res.status(500).send({
                    is_error: true,
                    message: "Error updating the account details."
                });
            });
    } else {
        // Throw error
        return res.status(400).send({ "is_error": true, "message": "Unauthorized Access!" });
    }
};

// send Contact Us query
exports.submitContactUsForm = async (req, res) => {
    // Validate request
    if (!(req.body.user_message)) {
        return res.status(400).send({ "is_error": true, "message": "Please provide your issue description." });
    }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    if (loggedInUser && loggedInUser.data && loggedInUser.data.user_id) {

        const user = await User.findOne({ _id: loggedInUser.data.user_id });
        // const textMsg = 'Dear Admin,' + req.body.user_message + ' Thanks , Team LifeStampR Support.';
        // const htmlTextMsg = '<h6> Dear Admin,</h6> Customer - <strong>' + user.first_name + ' ' + user.last_name + ',</strong> submitted below query: </br>' + req.body.user_message + ' </br> Thanks, </br> Team LifeStampR Support.'
        const emailData = {
            msg: {
                to: ['book@ctn.vacations'],
                from: 'book@ctn.vacations', // Use the email address or domain you verified above
                template_id: email_templates.CUSTOMER_SUPPORT_TEMPLATE_ID,
                dynamic_template_data:{
                  "first_name": user.first_name,
                  "last_name": user.last_name,
                  "user_email": "book@ctn.vacations",
                  "customer_message": req.body.user_message
                }
              },
            userDetails: user
        }
        // send Email to User with OTP
        emailCtrl.sendEmail(emailData);
        return res.send({ is_error: false, message: "Your message submitted successfully." });


    } else {
        // Throw error
        return res.status(400).send({ "is_error": true, "message": "Unauthorized Access!" });
    }

};

// Change my Password User
exports.changeMyPassword = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { current_pwd } = req.body;

        // Validate user input
        if (!(current_pwd)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide current Password." });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const user = await User.findOne({ _id: loggedInUser.data.user_id });
            console.log("found user", user);
            console.log(await bcrypt.compare(current_pwd, user.password))
            if (user && (await bcrypt.compare(current_pwd, user.password))) {
                // const user = await User.findOne({ email });

                const OTP = otpGenerator.generate(4, { digits: true, lowerCaseAlphabets: false, upperCaseAlphabets: false, specialChars: false });
                const expireTime = moment(new Date(), "DD-MM-YYYY hh:mm:ss")
                    .add(1, 'minutes')
                    .format('DD/MM/YYYY hh:mm:ss');

                const textMsg = 'Dear' + user.first_name + ' ' + user.last_name + ', Your OTP verification code is ' + OTP + ' Thanks , Team LifeStampR Support.';
                const htmlTextMsg = '<h6> Dear ' + user.first_name + ' ' + user.last_name + ',</h6><p> Your OTP verification code is </br> <strong>' + OTP + '</strong></p> </br> Thanks, </br> Team LifeStampR Support.'
                const emailData = {
                  
                    msg: {
                        to: [user.email],
                        from: 'book@ctn.vacations', // Use the email address or domain you verified above
                        template_id: email_templates.CHANGE_PASSWORD_EMIL_TEMPLATE_ID,
                        dynamic_template_data:{
                          "first_name": user.first_name,
                          "last_name": user.last_name,
                          "OTP": OTP
                        }
                      },
                    userDetails: user
                }

                const OTP_Payload = {
                    user_id: user._id,
                    email: user.email,
                    otpvalue: OTP
                };
                console.log("OTP Payload", OTP_Payload)
                let OTP_data = await OTPModel.findOneAndUpdate({ email: user.email, user_id: user._id }, OTP_Payload, {
                    new: true,
                    upsert: true // Make this update into an upsert
                });
                if (OTP_data) {
                    // send Email to User with OTP
                      emailCtrl.sendEmail(emailData);
                    return res.send({ is_error: false, data: { }, message: "OTP has been sent to email . Please verify the OTP." });

                } else {
                    return res.status(400).send({ "is_error": true, "message": "Error while change password. Please try after some time." });
                }
            } else {
                return res.status(400).send({ "is_error": true, "message": "Invalid Password. Please provide correct password." });
            }
        }
    } catch (err) {
        console.log(err);
    }

};

// Validate otp for change password.
exports.validateOtp = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { otp } = req.body;

        // Validate user input
        if (!(otp)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide OTP." });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const loggedinUserDetails = await User.findOne({ _id: loggedInUser.data.user_id });
            // Validate if otp exist in our database
            const user = await OTPModel.findOne({ 'email': loggedinUserDetails.email, 'otpvalue': req.body.otp });

            if (user && loggedinUserDetails) {

                // Find note and update it with the request body
                OTPModel.findByIdAndUpdate(user._id, { otp_verfied: true })
                    .then(otpData => {
                        if (!otpData) {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        return res.send({
                            is_error: false,
                            message: "OTP Verified successfully."
                        });
                    }).catch(err => {
                        if (err.kind === 'ObjectId') {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        return res.status(500).send({
                            is_error: true,
                            message: "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired."
                        });
                    });

            } else {
                return res.status(400).send({ "is_error": true, "message": "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired" });
            }
        }
    } catch (err) {
        console.log(err);
    }
    // Our logic ends here
};

// update user password
exports.updatePassword = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { newpassword } = req.body;

        // Validate user input
        if (!(newpassword)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide passowrd" });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const user = await User.findOne({ _id: loggedInUser.data.user_id });
            // Validate if otp exist in our database
            const OTPfound = await OTPModel.findOne({ 'email': user.email, 'otpvalue': req.body.otp });

            if (user && OTPfound && OTPfound.otp_verfied) {
                //Encrypt user password
                encryptedPassword = await bcrypt.hash(newpassword, 10);

                // delete otp entry from DB
                OTPModel.findByIdAndRemove(OTPfound._id)
                    .then(otpData => {
                        if (!otpData) {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        // Find user and update password
                        User.findByIdAndUpdate(user._id, {
                            password: encryptedPassword,
                        })
                            .then(userData => {
                                if (!userData) {
                                    return res.status(404).send({
                                        is_error: true,
                                        message: "Error while updating password."
                                    });
                                }
                                return res.send({
                                    is_error: false,
                                    message: "Password changed successfully."
                                });
                            }).catch(err => {
                                if (err.kind === 'ObjectId') {
                                    return res.status(404).send({
                                        is_error: true,
                                        message: "Error while updating Password."
                                    });
                                }
                                return res.status(500).send({
                                    is_error: true,
                                    message: "Error while updating Password."
                                });
                            });
                    }).catch(err => {
                        if (err.kind === 'ObjectId' || err.name === 'NotFound') {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        return res.status(500).send({
                            is_error: true,
                            message: "Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired."
                        });
                    });

            } else {
                res.status(400).send({ "is_error": true, "message": "Error while updating Password. Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired." });
            }
        }
    } catch (err) {
        console.log(err);
    }
    // Our logic ends here
};

// Update a user identified by the userId in the request
exports.changeProfilePic = async (req, res) => {

     //keeping track of all files uploaded
     var dbData = [];
     var imagePath ;
     req.pipe(req.busboy); // Pipe it trough busboy
     
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    const uploadPath = path.resolve("./public/assets/images/"+ loggedInUser.data.user_id + "/avatar/");
    fse.ensureDir(uploadPath); // Make sure that he upload path exits

    req.busboy.on('file', (fieldname, file, filename) => {
        const imageExtension = (filename.filename).split('.')[(filename.filename).split('.').length - 1];
        // 32756238461724837.png
        imageFileName = (loggedInUser.data.user_id)+'.'+imageExtension;
        console.log("fieldname", fieldname);
        console.log(`Upload of '${filename.filename}' started`);
        imagePath = '/static/assets/images/' + loggedInUser.data.user_id + '/avatar/' + imageFileName;
        // imagePath = '/static/assets/images/' + loggedInUser.data.user_id + '/avatar/' + ( loggedInUser.data.user_id)+'.jpg';
    
        dbData.push(path.join(uploadPath, imageFileName));
        // imageKeyData.push({key: fieldname, path:imagePath});
        // Create a write stream of the new file
        const fstream = fs.createWriteStream(path.join(uploadPath, imageFileName));
        // Pipe it trough
        file.pipe(fstream);

        // On finish of the upload
        fstream.on('close', () => {
            console.log(`Upload of '${imageFileName}' finished`);
        });
    });
    req.busboy.on('finish', function() {
        console.info('Busboy uploading finished of user controller!');
        // processImageUploadData(req, res , imagePath, dbData);
        // Find note and update it with the request body
   User.findByIdAndUpdate(loggedInUser.data.user_id, {
    profileImage: imagePath,
    }, { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found."
                });
            }
            return res.send({
                is_error: false,
                data: userData,
                message: "Profile Picture has been updated successfully."
            });
        }).catch(err => {
            dbData.forEach(item => {
                if (fs.existsSync(item)) {
                    fs.unlink(item, (err) => {
                        if (err) {
                            console.log(err);
                        }
                        console.log('deleted');
                    })
                }
            })
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the account details."
            });
        });
    });

};

// Change user status
exports.changeStatus = async (req, res) => {
    await User.findByIdAndUpdate(req.params.userId, {
        is_active: req.body.is_active
    }, { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "Invalid User Details."
                });
            }
            const notifyMsg = req.body.is_active ? 'Activated' : 'In-activated';
            res.send({'message': 'User Profile has been ' +' '+ notifyMsg });
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the user status."
            });
        });
};

// logged in Admin user details- Admin user
exports.myDetails = (req, res) => {
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    User.findById(loggedInUser.data.user_id).select("-password -__v")
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + loggedInUser.data.user_id
                });
            }
            res.send({ is_error: false, data: userData });
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + loggedInUser.data.user_id
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error retrieving user with id " + loggedInUser.data.user_id
            });
        });
};

// send Contact Us query
exports.changeAdminPassword = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { currentPwd, newPwd } = req.body;

        // Validate user input
        if (!(currentPwd && newPwd)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide current Password." });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const user = await User.findOne({ _id: loggedInUser.data.user_id });
            
            if (user && (await bcrypt.compare(currentPwd, user.password))) {
                //Encrypt user password
                encryptedPassword = await bcrypt.hash(newPwd, 10);
                // Find user and update password
                User.findByIdAndUpdate(user._id, {
                    password: encryptedPassword,
                })
                    .then(userData => {
                        if (!userData) {
                            return res.status(404).send({
                                is_error: true,
                                message: "Error while updating password."
                            });
                        }
                        return res.send({
                            is_error: false,
                            message: "Password changed successfully."
                        });
                    }).catch(err => {
                        if (err.kind === 'ObjectId') {
                            return res.status(404).send({
                                is_error: true,
                                message: "Error while updating Password."
                            });
                        }
                        return res.status(500).send({
                            is_error: true,
                            message: "Error while updating Password."
                        });
                    });
            } else {
                return res.status(400).send({ "is_error": true, "message": "Invalid Password. Please provide your correct current password." });
            }
        }
    } catch (err) {
        console.log(err);
    }

};

// Find a single user with a email to use in share event feature
exports.findUserByEmail = async (req, res) => {
    console.log("req.body", req.body)
    // Validate user input
    if (!req.body.email) {
        return res.status(400).send({ "is_error": true, "message": "Please provide user email to search." });
    }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    const query = {$or:[  {'first_name':{ $regex: '.*' + req.body.email+ '.*', $options: 'i' }},
            {'last_name':{ $regex: '.*' + req.body.email + '.*', $options: 'i' }},
            {'email': req.body.email}], 
            is_admin: false, is_active: true, _id: { $ne: loggedInUser.data.user_id } }
        // await User.find({is_admin: false, is_active: true, email: req.body.email ,  _id: { $ne: loggedInUser.data.user_id }}).select("-password -__v -events -templates -createdAt -updatedAt -role -is_active -is_admin")
        await User.find(query).select("-password -__v -events -templates -createdAt -updatedAt -role -is_active -is_admin")
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "No user found for :  " + req.body.email
                });
            }
            res.send({ is_error: false, data: userData});
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "No user found for : " + req.body.email
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error in searching user."
            });
        });
};

processImageUploadData = async (req,res, imagePath , dbData) =>{
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
   // Find note and update it with the request body
   User.findByIdAndUpdate(loggedInUser.data.user_id, {
    profileImage: imagePath,
    }, { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found."
                });
            }
            return res.send({
                is_error: false,
                data: userData,
                message: "Profile Picture has been updated successfully."
            });
        }).catch(err => {
            dbData.forEach(item => {
                if (fs.existsSync(item)) {
                    fs.unlink(item, (err) => {
                        if (err) {
                            console.log(err);
                        }
                        console.log('deleted');
                    })
                }
            })
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the account details."
            });
        });
}