const templateModel = require('../model/template');
const User = require('../model/user');
const eventModel = require('../model/events');

const helper = require('../middleware/_helper');

// Create and Save a new template - only form app users to create custom templates
exports.create = async (req, res) => {
    console.log("event Body data", req.body)
    // Validate request
    if (!(req.body.template_name && req.body.components)) {
        return res.status(400).send({ is_error: true , message: "Template Name & Template Fields all input is required" });
      }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    // check if template already exist and created by same user
    // Validate if oldTemplate exist in our database
    const oldTemplate = await templateModel.findOne({ 'template_name': req.body.template_name ,'created_by': loggedInUser.data.user_id });

    if (oldTemplate) {
    return res.status(409).send({"is_error": true , "message":  req.body.template_name + "  Template already created with same name." });
    }
    const payload = {
        "template_name": req.body.template_name,
        "components": req.body.components,
        "price": 0.0,
        "type": "USER_DEFINED",
        "created_by":loggedInUser.data.user_id,
        "linkedTemplates": req.body.linkedTemplates
      };
      

    // Create a event
    const event = new templateModel(payload);

    // Save event in the database
    event.save()
    .then(data => {
        res.send({is_error : false, data:data});
    }).catch(err => {
        return res.status(500).send({
            is_error : true,
            message: err.message || "Some error occurred while creating the Template."
        });
    });
};

// Only used by ADMIN User.
exports.createDefault = async (req, res) => {
    console.log("event Body data", req.body)
    // Validate request
    if (!(req.body.template_name && req.body.components)) {
        return res.status(400).send({ is_error: true , message: "Template Name & Template Fields all input is required" });
      }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    // check if template already exist and created by same user
    // Validate if oldTemplate exist in our database
    const oldTemplate = await templateModel.findOne({ 'template_name': req.body.template_name ,'created_by': loggedInUser.data.user_id });

    if (oldTemplate) {
    return res.status(409).send({"is_error": true , "message":  req.body.template_name + " Template already created with same name." });
    }

    const payload = {
        "template_name": req.body.template_name,
        "components": req.body.components,
        "price": req.body.price,
        "type": req.body.type,
        "created_by":loggedInUser.data.user_id,
        "linkedTemplates": req.body.linkedTemplates
      };
      

    // Create a event
    const event = new templateModel(payload);

    // Save event in the database
    event.save()
    .then(data => {
        res.send({is_error : false, data:data});
    }).catch(err => {
        return res.status(500).send({
            is_error : true,
            message: err.message || "Some error occurred while creating the Template."
        });
    });
};
// Retrieve and return all templates from the database.
exports.findAll = async (req, res) => {
    try {
       const templates = await templateModel.find({})
        .populate({
            path:'created_by',
            select:['first_name', 'last_name', '_id','email']
        })
        .populate({
            path: 'linkedTemplates',
            // Get child templates of templates - populate the 'Child template' array for every Template
            populate: { path: 'linkedTemplates' }
        })
        .sort([["template_name",1]]);
        // .sort([["updatedAt",-1]]);
        res.send({is_error : false, data: templates});

    } catch (error) {
        console.log(error);
        return res.send({is_error : true, message:"An error occured : " + error});
    }
  };

// Find a single templates with a id
exports.findOne = (req, res) => {
    templateModel.findById(req.params.id).populate({
        path: 'linkedTemplates',
        // Get child templates of templates - populate the 'Child template' array for every Template
        populate: { path: 'linkedTemplates' }
    })
    .then(response => {
        if(!response) {
            return res.status(404).send({
                is_error : true,
                message: "Template not found with id " + req.params.id
            });            
        }
        res.send({is_error : false, data: response });
    }).catch(err => {
        if(err.kind === 'ObjectId') {
            return res.status(404).send({
                is_error : true,
                message: "Template not found with id " + req.params.id
            });                
        }
        return res.status(500).send({
            is_error : true,
            message: "Error retrieving Template with id " + req.params.id
        });
    });
};

// Update a Template identified by the id in the request
exports.update = async (req, res) => {
    console.log("update Template", req.body)
    // Validate request
    if (!(req.body.template_name && req.body.components)) {
        return res.status(400).send({ is_error: true , message: "All input is required" });
      }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    // check if template already exist and created by same user
    // Validate if oldTemplate exist in our database
    const oldTemplate = await templateModel.findOne({  _id: { $ne: req.params.id } , 'template_name': req.body.template_name ,'created_by': loggedInUser.data.user_id });

    if (oldTemplate) {
    return res.status(409).send({"is_error": true , "message":  req.body.template_name + " Template already created with same name." });
    }
    const payload = {
        "template_name": req.body.template_name,
        "components": req.body.components,
        "price": req.body.price,
        "type": req.body.type,
        "linkedTemplates": req.body.linkedTemplates
      };

    // Find Template and update it with the request body
    templateModel.findByIdAndUpdate(req.params.id, payload , {new: true})
    .then(response => {
        if(!response) {
            return res.status(404).send({
                is_error : true,
                message: "Template not found with id " + req.params.id
            });
        }
        res.send({is_error : false, data: response});
    }).catch(err => {
        if(err.kind === 'ObjectId') {
            return res.status(404).send({
                is_error : true,
                message: "Template not found with id " + req.params.id
            });                
        }
        return res.status(500).send({
            is_error : true,
            message: "Error updating Template with id " + err
        });
    });
};

// Delete a Template with the specified id in the request
exports.delete = async (req, res) => {
    // check if any event(s) associated with this template then throw error
   try{
        const associatedEvents =  await eventModel.find({'template_id': req.params.id}).select("-components -location -tags -reminder -frequency -persontags -duration -category -subcategory -childEvents -repeats -reminderAt -start_date -end_date -start_time -end_time -description -sharedList -createdAt -updatedAt -parent_id -is_active -is_completed")
        console.log("associatedEvents", associatedEvents)
        if(associatedEvents && associatedEvents.length >0 ){
            var eventsNameArr = associatedEvents.map(elem => {
                return elem.name ;
            })
            return res.status(400).send({"is_error": true , "message": "This template cannot be deleted since you have events using it. Delete events based on this template first then delete the template."});
        }
    }
    catch (error) {
        console.log(error);
        return res.send({"is_error": true , "message": "An error occured : " + error});
    }
    console.log("everthung good go ahead with delete template now")
    templateModel.findByIdAndRemove(req.params.id)
    .then(response => {
        if(!response) {
            return res.status(404).send({
                is_error : true,
                message: "Template not found with id " + req.params.id
            });
        }
        res.send({is_error : false, message: "Template deleted successfully!"});
    }).catch(err => {
        if(err.kind === 'ObjectId' || err.name === 'NotFound') {
            return res.status(404).send({
                is_error : true,
                message: "Template not found with id " + req.params.id
            });                
        }
        return res.status(500).send({
            is_error : true,
            message: "Could not delete Template with id " + req.params.id
        });
    });
};

// Retrieve and return templates based on type ( default/ paid ) from the database.
exports.getSystemTemplates = async (req, res) => {
    try {
        console.log("req  params", req.params);
        console.log("req query params", req.query);
        const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
        const userfavtemplates = await User.findOne({  _id: loggedInUser.data.user_id });

        if(!req.params.templateType){
        return res.send({is_error : true, message:"Please provide the template type."});
        }
        const query = req.params.templateType == 'ALL' ? {$or:[ {'type':'FREE'},{'type':'PAID'}], 'is_active':true} : {'type':req.params.templateType, 'is_active':true};
        
        if(req.query.limit){
        const limitCount = parseInt(req.query.limit); 
        const templates = await templateModel.find(query).sort([["template_name",1]]).limit(limitCount);
        console.log("response data count", templates.length)
            // Get all Favourite templates of user
            if(userfavtemplates && userfavtemplates.fav_templates && userfavtemplates.fav_templates.length > 0){
                console.log("user fav template", userfavtemplates.fav_templates)
                // Assigned is_fav to all templates.
                templates.forEach( (record , index ) => {
                    if((userfavtemplates.fav_templates).includes(record._id)){
                        Object.assign(templates[index], {is_favourite: true});
                    }else{
                        Object.assign(templates[index], {is_favourite: false});
                    }
                })
                res.send({is_error : false, data: templates});
            }else{
                res.send({is_error : false, data: templates});
            }
        }
        else{
            const templates = await templateModel.find(query).sort([["template_name",1]]);
             console.log("response data count", templates.length)
            // Get all Favourite templates of user
            if(userfavtemplates && userfavtemplates.fav_templates && userfavtemplates.fav_templates.length > 0){
                console.log("user fav template", userfavtemplates.fav_templates)
                // Assigned is_fav to all templates.
                templates.forEach( (record , index ) => {
                    if((userfavtemplates.fav_templates).includes(record._id)){
                        Object.assign(templates[index], {is_favourite: true});
                    }else{
                        Object.assign(templates[index], {is_favourite: false});
                    }
                })
                res.send({is_error : false, data: templates});
            }else{
                res.send({is_error : false, data: templates});
            }
        }
        
    } catch (error) {
        console.log(error);
        return res.send({is_error : true, message:"An error occured : " + error});
    }
  };

  // Retrieve and return all my templates + default (Free) templates from the database.
exports.getMyTemplates = async (req, res) => {
    try {
        const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
        const userfavtemplates = await User.findOne({  _id: loggedInUser.data.user_id });
        
        if (req.query.limit) {
            const limitCount = parseInt(req.query.limit);
            const templates = await templateModel.find( {$or:[ {'type':'USER_DEFINED'}],'created_by':loggedInUser.data.user_id}).sort([["template_name",1]]).limit(limitCount);
            console.log("response data count", templates.length);
            // Get all Favourite templates of user
            if(userfavtemplates && userfavtemplates.fav_templates && userfavtemplates.fav_templates.length > 0){
                console.log("user fav template", userfavtemplates.fav_templates)
                // Assigned is_fav to all templates.
                templates.forEach( (record , index ) => {
                    if((userfavtemplates.fav_templates).includes(record._id)){
                        Object.assign(templates[index], {is_favourite: true});
                    }else{
                        Object.assign(templates[index], {is_favourite: false});
                    }
                })
                res.send({is_error : false, data: templates});
            }else{
                res.send({is_error : false, data: templates});
            }
        }
        else {
            const templates = await templateModel.find( {$or:[ {'type':'USER_DEFINED'}],'created_by':loggedInUser.data.user_id}).sort([["template_name",1]]);
            // Get all Favourite templates of user
            if(userfavtemplates && userfavtemplates.fav_templates && userfavtemplates.fav_templates.length > 0){
                console.log("user fav template", userfavtemplates.fav_templates)
                // Assigned is_fav to all templates.
                templates.forEach( (record , index ) => {
                    if((userfavtemplates.fav_templates).includes(record._id)){
                        Object.assign(templates[index], {is_favourite: true});
                    }else{
                        Object.assign(templates[index], {is_favourite: false});
                    }
                })
                res.send({is_error : false, data: templates});
            }else{
                res.send({is_error : false, data: templates});
            }
        }
    } catch (error) {
        console.log(error);
        return res.send({is_error : true, message:"An error occured : " + error});
    }
  };

  // Retrieve and return all my templates + default (Free) templates from the database.
exports.getUserAllTemplates = async (req, res) => {
try {
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    const templates = await templateModel.find( {'type':'USER_DEFINED' ,'created_by':loggedInUser.data.user_id}).sort([["template_name",-1]]);
    const Freetemplates = await templateModel.find( {'type':'FREE','is_active':true}).sort([["template_name",-1]]);
    const premiumTemplates = await templateModel.find( {'type':'PREMIUM','is_active':true}).sort([["template_name",-1]]);
    const PurchasedTemplates = await User.findOne({_id: loggedInUser.data.user_id})
        .populate({
            path:'templates'
        }).sort([["template_name",-1]]);
    const paidTemplates = PurchasedTemplates.templates ;
    const allTemplates = [...templates ,...paidTemplates, ...Freetemplates, ...premiumTemplates];
    allTemplates.sort((a, b) => a.template_name.toLowerCase() > b.template_name.toLowerCase() ? 1 : -1);
    // Get all Favourite templates of user
    const userfavtemplates = await User.findOne({  _id: loggedInUser.data.user_id });
    if(userfavtemplates && userfavtemplates.fav_templates && userfavtemplates.fav_templates.length > 0){
        console.log("user fav template", userfavtemplates.fav_templates)
        // Assigned is_fav to all templates.
         allTemplates.forEach( (record , index ) => {
            if((userfavtemplates.fav_templates).includes(record._id)){
                Object.assign(allTemplates[index], {is_favourite: true});
            }else{
                Object.assign(allTemplates[index], {is_favourite: false});
            }
        })
        res.send({is_error : false, data: allTemplates});
    }else{
        res.send({is_error : false, data: allTemplates});
    }

} catch (error) {
    console.log(error);
    return res.send({is_error : true, message:"An error occured : " + error});
}
};

// Change Template status
exports.changeStatus = async (req, res) => {
    await templateModel.findByIdAndUpdate(req.params.userId, {
        is_active: req.body.is_active
    }, { new: true })
        .then(templateData => {
            if (!templateData) {
                return res.status(404).send({
                    is_error: true,
                    message: "Invalid Template Details."
                });
            }
            const notifyMsg = req.body.is_active ? 'Activated' : 'In-activated';
            res.send({'message': 'Template has been ' +' '+ notifyMsg });
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "Template details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the template status."
            });
        });
};

// Retrieve and return all templates from the database.
exports.findAllWithStats = async (req, res) => {
    try {
        const templates = await templateModel.find({})
        .populate({
            path:'created_by',
            select:['first_name', 'last_name', '_id','email']
        })
        .populate({
            path: 'linkedTemplates',
            // Get child templates of templates - populate the 'Child template' array for every Template
            populate: { path: 'linkedTemplates' }
        })
        .sort([["updatedAt",-1]]);

     
        let tempData = [];
        let promiseList = await templates.map(async (item,i)=> 
        eventModel.distinct('created_by', {'template_id': item._id}).exec() 
        );
        Promise.all(promiseList).then(async (uniqueUserList)=> {
                templates.forEach(async (element,index) => {
                    var templateObj =  element.toObject();
                    templateObj.usersUsingTemplateCount = uniqueUserList[index].length
                    tempData.push(templateObj);
                });
                const aggregatorOpts = [{
                    $group: {
                      _id: "$template_id",
                      count: { $sum: 1 }
                    }
                  }]
                const eventsUsingTemplateCount =  await eventModel.aggregate(aggregatorOpts).exec();
                res.send({is_error : false, data: tempData , eventsUsingTemplateCount:eventsUsingTemplateCount});
                
            });

    } catch (error) {
        console.log(error);
        return res.send({is_error : true, message:"An error occured : " + error});
    }
  };

  exports.addToFavourite = async (req, res) => {
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);

     // Validate request
    if (!req.body.templateId) {
        return res.status(400).send({ is_error: true , message: "Please select any Template." });
    }
    const templateData = await templateModel.findOne({  _id: req.body.templateId });

    if (!templateData) {
    return res.status(409).send({"is_error": true , "message":  "No Template available with the provided reference id." });
    }
      await User.findByIdAndUpdate(
        loggedInUser.data.user_id, 
        { $push: { fav_templates: templateData._id } }, 
        { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found in our record."
                });
            }

          res.send({is_error : false, message:"Template Added in your Favourite List."});
  
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found in our record."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error while updating the template details to user's account."
            });
        });

}

exports.removeFromFavourite = async (req, res) => {
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);

     // Validate request
    if (!req.body.templateId) {
        return res.status(400).send({ is_error: true , message: "Please select any Template." });
    }
    const templateData = await templateModel.findOne({  _id: req.body.templateId });

    if (!templateData) {
    return res.status(409).send({"is_error": true , "message":  "No Template available with the provided reference id." });
    }

      await User.findByIdAndUpdate(
        loggedInUser.data.user_id, 
        {$pull: {fav_templates: req.body.templateId}}, 
        { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found in our record."
                });
            }

          res.send({is_error : false, message:"Template Removed from your Favourite List."});
  
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found in our record."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error while updating the template details to user's account."
            });
        });

}

// Retrieve and return all templates for child dropdown from the database.
exports.findAllTemplatesForChildDropDown = async (req, res) => {
    try {
       const templates = await templateModel.find({type:{$ne:'USER_DEFINED'}})
        .populate({
            path:'created_by',
            select:['first_name', 'last_name', '_id','email']
        })
        .populate({
            path: 'linkedTemplates',
            // Get child templates of templates - populate the 'Child template' array for every Template
            populate: { path: 'linkedTemplates' }
        })
        .sort([["template_name",1]]);
        // .sort([["updatedAt",-1]]);
        res.send({is_error : false, data: templates});

    } catch (error) {
        console.log(error);
        return res.send({is_error : true, message:"An error occured : " + error});
    }
  };